<?php
// File: klinik-topsis/app/core/Database.php
// Deskripsi: Kelas untuk mengelola koneksi database menggunakan PDO (PHP Data Objects).
// Menggunakan pola Singleton untuk memastikan hanya ada satu koneksi database yang dibuat.

class Database {
    // Properti untuk koneksi database
    private $host = localhost;
    private $user = qktjehko_kliniktopsis;
    private $pass = Amigemash12345;
    private $db_name = qktjehko_kliniktopsis;

    // Properti untuk PDO
    private $dbh; // Database Handler
    private $stmt; // Statement

    private static $instance = null;

    /**
     * Constructor dibuat private untuk mencegah pembuatan instance langsung
     * dari luar kelas (pola Singleton).
     */
    private function __construct() {
        // Data Source Name (DSN) untuk koneksi PDO
        $dsn = 'mysql:host=' . $this->host . ';dbname=' . $this->db_name;

        // Opsi untuk koneksi PDO
        $options = [
            PDO::ATTR_PERSISTENT => true, // Menjaga koneksi tetap terbuka
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION // Mode error untuk menampilkan exception
        ];

        try {
            // Membuat instance PDO baru
            $this->dbh = new PDO($dsn, $this->user, $this->pass, $options);
        } catch (PDOException $e) {
            // Jika koneksi gagal, hentikan aplikasi dan tampilkan pesan error
            die($e->getMessage());
        }
    }

    /**
     * Metode static untuk mendapatkan instance dari kelas Database (Singleton).
     * @return Database instance
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    // Mencegah kloning instance (Singleton)
    private function __clone() {}

    // Mencegah unserialize instance (Singleton)
    public function __wakeup() {}

    /**
     * Menyiapkan query SQL.
     * @param string $query Query SQL yang akan dieksekusi.
     */
    public function query($query) {
        $this->stmt = $this->dbh->prepare($query);
    }

    /**
     * Melakukan binding parameter ke query.
     * @param mixed $param Parameter (e.g., :nama).
     * @param mixed $value Nilai yang akan di-bind.
     * @param mixed $type Tipe data dari parameter (e.g., PDO::PARAM_STR).
     */
    public function bind($param, $value, $type = null) {
        if (is_null($type)) {
            switch (true) {
                case is_int($value):
                    $type = PDO::PARAM_INT;
                    break;
                case is_bool($value):
                    $type = PDO::PARAM_BOOL;
                    break;
                case is_null($value):
                    $type = PDO::PARAM_NULL;
                    break;
                default:
                    $type = PDO::PARAM_STR;
            }
        }
        $this->stmt->bindValue($param, $value, $type);
    }

    /**
     * Mengeksekusi statement yang sudah disiapkan.
     * @return bool True jika berhasil, false jika gagal.
     */
    public function execute() {
        return $this->stmt->execute();
    }

    /**
     * Mengembalikan hasil query sebagai array of objects.
     * @return array
     */
    public function resultSet() {
        $this->execute();
        return $this->stmt->fetchAll(PDO::FETCH_OBJ);
    }

    /**
     * Mengembalikan satu baris hasil query sebagai object.
     * @return object
     */
    public function single() {
        $this->execute();
        return $this->stmt->fetch(PDO::FETCH_OBJ);
    }

    /**
     * Mengembalikan jumlah baris yang terpengaruh oleh query (INSERT, UPDATE, DELETE).
     * @return int
     */
    public function rowCount() {
        return $this->stmt->rowCount();
    }
}
