<?php
// File: klinik-topsis/app/models/Topsis_model.php
// Deskripsi: Model untuk melakukan seluruh perhitungan metode TOPSIS.

class Topsis_model {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Metode utama untuk menjalankan seluruh proses perhitungan TOPSIS.
     * @param int $id_bidan ID bidan yang sedang login.
     * @return array Hasil perhitungan lengkap atau array kosong jika tidak ada data.
     */
    public function hitung($id_bidan) {
        // --- LANGKAH 1: Mengambil Data Mentah ---
        $periode = date('Y-m-d');
        
        // Matriks Keputusan (Alternatif x Kriteria)
        $matriks_keputusan = $this->getMatrixKeputusan($id_bidan, $periode);
        if (empty($matriks_keputusan)) {
            return ['error' => 'Belum ada data penilaian untuk periode ini.'];
        }

        // Bobot dan Tipe Kriteria
        $kriteria_data = $this->getKriteriaData();
        $bobot = $kriteria_data['bobot'];
        $tipe = $kriteria_data['tipe'];
        $nama_pasien = $this->getNamaPasien($id_bidan);

        // --- LANGKAH 2: Normalisasi Matriks Keputusan (R) ---
        $pembagi = $this->hitungPembagi($matriks_keputusan);
        $matriks_normalisasi = $this->normalisasi($matriks_keputusan, $pembagi);
        
        // --- LANGKAH 3: Normalisasi Terbobot (Y) ---
        $matriks_terbobot = $this->normalisasiTerbobot($matriks_normalisasi, $bobot);

        // --- LANGKAH 4: Menentukan Solusi Ideal Positif (A+) & Negatif (A-) ---
        $solusi_ideal = $this->getSolusiIdeal($matriks_terbobot, $tipe);
        $solusi_ideal_positif = $solusi_ideal['A+'];
        $solusi_ideal_negatif = $solusi_ideal['A-'];

        // --- LANGKAH 5: Menghitung Jarak ke Solusi Ideal (D+ & D-) ---
        $jarak = $this->getJarakSolusiIdeal($matriks_terbobot, $solusi_ideal_positif, $solusi_ideal_negatif);
        $jarak_positif = $jarak['D+'];
        $jarak_negatif = $jarak['D-'];

        // --- LANGKAH 6: Menghitung Nilai Preferensi (V) dan Melakukan Perankingan ---
        $hasil_akhir = $this->getNilaiPreferensi($jarak_positif, $jarak_negatif, $nama_pasien);

        // Mengembalikan semua hasil perhitungan untuk ditampilkan di view
        return [
            'matriks_keputusan' => $matriks_keputusan,
            'pembagi' => $pembagi,
            'matriks_normalisasi' => $matriks_normalisasi,
            'bobot' => $bobot,
            'matriks_terbobot' => $matriks_terbobot,
            'solusi_ideal_positif' => $solusi_ideal_positif,
            'solusi_ideal_negatif' => $solusi_ideal_negatif,
            'jarak_positif' => $jarak_positif,
            'jarak_negatif' => $jarak_negatif,
            'hasil_akhir' => $hasil_akhir,
            'nama_pasien' => $nama_pasien
        ];
    }
    
    // --- Metode-metode Pembantu ---

    private function getMatrixKeputusan($id_bidan, $periode) {
        $this->db->query("
            SELECT p.id_pasien, p.id_kriteria, p.nilai
            FROM penilaian p
            JOIN pasien ps ON p.id_pasien = ps.id
            WHERE ps.id_bidan = :id_bidan AND p.periode_penilaian = :periode
            ORDER BY p.id_pasien, p.id_kriteria
        ");
        $this->db->bind(':id_bidan', $id_bidan);
        $this->db->bind(':periode', $periode);
        $penilaian = $this->db->resultSet();

        $matriks = [];
        foreach ($penilaian as $row) {
            $matriks[$row->id_pasien][$row->id_kriteria] = $row->nilai;
        }
        return $matriks;
    }
    
    private function getKriteriaData() {
        $this->db->query("SELECT id, bobot, tipe FROM kriteria ORDER BY id");
        $kriteria = $this->db->resultSet();
        $data = ['bobot' => [], 'tipe' => []];
        foreach($kriteria as $row) {
            $data['bobot'][$row->id] = $row->bobot;
            $data['tipe'][$row->id] = $row->tipe;
        }
        return $data;
    }

    private function getNamaPasien($id_bidan) {
        $this->db->query("SELECT id, nama_ibu FROM pasien WHERE id_bidan = :id_bidan");
        $this->db->bind(':id_bidan', $id_bidan);
        $pasien = $this->db->resultSet();
        $nama = [];
        foreach($pasien as $row) {
            $nama[$row->id] = $row->nama_ibu;
        }
        return $nama;
    }

    private function hitungPembagi($matriks) {
        $pembagi = [];
        foreach (current($matriks) as $id_kriteria => $nilai) {
            $jumlah_kuadrat = 0;
            foreach ($matriks as $id_pasien => $penilaian) {
                $jumlah_kuadrat += pow($penilaian[$id_kriteria], 2);
            }
            $pembagi[$id_kriteria] = sqrt($jumlah_kuadrat);
        }
        return $pembagi;
    }
    
    private function normalisasi($matriks, $pembagi) {
        $matriks_normal = [];
        foreach ($matriks as $id_pasien => $penilaian) {
            foreach ($penilaian as $id_kriteria => $nilai) {
                if ($pembagi[$id_kriteria] == 0) {
                     $matriks_normal[$id_pasien][$id_kriteria] = 0;
                } else {
                     $matriks_normal[$id_pasien][$id_kriteria] = $nilai / $pembagi[$id_kriteria];
                }
            }
        }
        return $matriks_normal;
    }
    
    private function normalisasiTerbobot($matriks_normal, $bobot) {
        $matriks_terbobot = [];
        foreach ($matriks_normal as $id_pasien => $penilaian) {
            foreach ($penilaian as $id_kriteria => $nilai) {
                $matriks_terbobot[$id_pasien][$id_kriteria] = $nilai * $bobot[$id_kriteria];
            }
        }
        return $matriks_terbobot;
    }

    private function getSolusiIdeal($matriks_terbobot, $tipe) {
        $solusi_ideal_positif = [];
        $solusi_ideal_negatif = [];
        foreach (current($matriks_terbobot) as $id_kriteria => $nilai) {
            $kolom = array_column($matriks_terbobot, $id_kriteria);
            if ($tipe[$id_kriteria] == 'benefit') {
                $solusi_ideal_positif[$id_kriteria] = max($kolom);
                $solusi_ideal_negatif[$id_kriteria] = min($kolom);
            } else { // 'cost'
                $solusi_ideal_positif[$id_kriteria] = min($kolom);
                $solusi_ideal_negatif[$id_kriteria] = max($kolom);
            }
        }
        return ['A+' => $solusi_ideal_positif, 'A-' => $solusi_ideal_negatif];
    }
    
    private function getJarakSolusiIdeal($matriks_terbobot, $positif, $negatif) {
        $jarak_positif = [];
        $jarak_negatif = [];
        foreach ($matriks_terbobot as $id_pasien => $penilaian) {
            $jarak_p = 0;
            $jarak_n = 0;
            foreach ($penilaian as $id_kriteria => $nilai) {
                $jarak_p += pow($nilai - $positif[$id_kriteria], 2);
                $jarak_n += pow($nilai - $negatif[$id_kriteria], 2);
            }
            $jarak_positif[$id_pasien] = sqrt($jarak_p);
            $jarak_negatif[$id_pasien] = sqrt($jarak_n);
        }
        return ['D+' => $jarak_positif, 'D-' => $jarak_negatif];
    }

    private function getNilaiPreferensi($jarak_positif, $jarak_negatif, $nama_pasien) {
        $preferensi = [];
        foreach ($jarak_negatif as $id_pasien => $d_neg) {
            $d_pos = $jarak_positif[$id_pasien];
            $v = ($d_pos + $d_neg == 0) ? 0 : $d_neg / ($d_pos + $d_neg);
            $preferensi[] = [
                'id_pasien' => $id_pasien,
                'nama_pasien' => $nama_pasien[$id_pasien] ?? 'Nama Tidak Ditemukan',
                'nilai_v' => $v
            ];
        }
        // Urutkan hasilnya dari nilai V tertinggi ke terendah
        usort($preferensi, function($a, $b) {
            return $b['nilai_v'] <=> $a['nilai_v'];
        });
        return $preferensi;
    }
    
    public function getTopPrioritas($id_bidan, $limit = 3) {
        // Kita panggil fungsi hitung() yang sudah ada
        $hasil_lengkap = $this->hitung($id_bidan);

        // Jika ada error (misal belum ada penilaian), kembalikan array kosong
        if (isset($hasil_lengkap['error'])) {
            return [];
        }

        // Ambil hanya sejumlah $limit dari hasil akhir
        return array_slice($hasil_lengkap['hasil_akhir'], 0, $limit);
    }
}
